/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <cmath>
#include <limits>
#include <stdint.h>
#include <utility>
#include <algorithm>
#include <cfloat>
#include <iterator>

#include "Matrix.h"

using namespace LayerManagerCalibration;
using namespace std;

Matrix::Matrix()
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            data[i][j] = 0.0;
        }
    }
}

Matrix::~Matrix()
{

}

Matrix& Matrix::operator=(const Matrix& rhs)
{
    if (this != &rhs)
    {
        for (int i = 0; i < MATRIX_ROWS; ++i)
        {
            for (int j = 0; j < MATRIX_COLUMNS; ++j)
            {
                data[i][j] = rhs.getData()[i][j];
            }
        }
    }

    return *this;
}

Matrix& Matrix::operator=(const Data& rhs)
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            data[i][j] = rhs[i][j];
        }
    }

    return *this;
}

Matrix Matrix::operator*(const Matrix& rhs)
{
    Data result;

    for (int i = 0; i < MATRIX_ROWS; i++)
    {
        for (int j = 0; j < MATRIX_COLUMNS; j++)
        {
            result[i][j] = ((data[i][0] * rhs.data[0][j])
                             + (data[i][1] * rhs.data[1][j])
                             + (data[i][2] * rhs.data[2][j]));
        }
    }

    return Matrix(result);
}

Matrix Matrix::operator*(const double rhs)
{
    Data resultData;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            resultData[i][j] = data[i][j] * rhs;
        }
    }

    return Matrix(resultData);
}

vector3 Matrix::operator*(const vector3& rhs)
{
    vector3 result;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        result.value[i] = 0.0;

        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            result.value[i] = result.value[i] + (data[i][j] * rhs.value[j]);
        }
    }

    return result;
}

bool Matrix::operator==(const Matrix& rhs)
{
    const Data& b = rhs.getData();
    const Data& a = this->getData();

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            if (!(fabs(a[i][j] - b[i][j])
               <= (max(fabs(a[i][j]), fabs(b[i][j]))
               * numeric_limits<double>::epsilon())))
            {
                return false;
            }
        }
    }
    return true;
}


double Matrix::calculateDeterminant()
{
    return ((data[0][0] * data[1][1] * data[2][2])
            + (data[0][1] * data[1][2] * data[2][0])
            + (data[0][2] * data[1][0] * data[2][1]))
          -
           ((data[0][2] * data[1][1] * data[2][0])
            + (data[0][1] * data[1][0] * data[2][2])
            + (data[0][0] * data[1][2] * data[2][1]));
}


void Matrix::transposeMatrix()
{
    Data matrixData;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            matrixData[i][j] = data[j][i];
        }
    }

    *this = matrixData;
}

void Matrix::adjugateMatrix()
{
    Data cofactor;

    // Create Cofactor of matrix
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            cofactor[i][j] = (data[(i + 1) % 3][(j + 1) % 3]
                              * data[(i + 2) % 3][(j + 2) % 3])
                              - (data[(i + 1) % 3][(j + 2) % 3]
                              * data[(i + 2) % 3][(j + 1) % 3]);
        }
    }

    // Transpose result and set
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            data[i][j] = cofactor[j][i];
        }
    }
}


Matrix::Matrix(const Data& matrixVal)
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            data[i][j] = matrixVal[i][j];
        }
    }
}

Matrix::Matrix(const Matrix& other)
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            data[i][j] = other.getData()[i][j];
        }
    }
}

std::ostream& LayerManagerCalibration::operator<<(std::ostream& os, const Matrix& rhs)
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            if (j != (MATRIX_COLUMNS - 1))
                os << rhs.getData()[i][j] << ", ";
            else
                os << rhs.getData()[i][j];
        }

        os << endl;
    }

    return os;
}

std::istream& LayerManagerCalibration::operator>>(std::istream& is, Matrix& rhs)
{
    LayerManagerCalibration::Matrix::Data matrixData;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            is >> matrixData[i][j];
        }
    }

    rhs.setData(matrixData);

    return is;
}
